﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System.Collections.Generic;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.MappingService.Interface;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public abstract class MapperRelatedBase : MapperBase, IEntityChild
    {
        private List<Entity> _modifyEntityList;
        
        protected List<Entity> ModifyEntityList => _modifyEntityList ?? (_modifyEntityList = new List<Entity>());

        protected MapperRelatedBase(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public abstract Task<Entity> MapInsert<T>(T entity, Entity parent);

        public abstract Task<Entity> MapUpdate<T>(T entity, Entity parent);

        public abstract void AddChildrenToProvider(IList<Entity> entities, Entity parent);

        public abstract IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent);

        public async Task<IEnumerable<Entity>> InsertMapperEntity<T>(
            IEnumerable<T> schemaList, 
            Entity parent, 
            bool skipProviderRelationship = false,
            bool forVaNetwork = false)
        {
            ForVaNetwork = forVaNetwork;

            var list = new List<Entity>();
            if (schemaList != null)
            {
                foreach (var item in schemaList)
                {
                    var child = await MapInsert(item, parent);
                    if (child != null)
                    {
                        list.Add(child);
                        ModifyEntityList.Add(child);
                    }
                }

                if (!skipProviderRelationship && ModifyEntityList.Count > 0 && parent.GetType() == typeof(Account)) AddChildrenToProvider(list, parent);
            }
            return list;
        }

        public async Task<IEnumerable<Entity>> UpdateMapperEntity<T>(
            IEnumerable<T> schemaList, 
            Entity parent, 
            bool skipProviderRelationship = false,
            bool forVaNetwork = false)
        {
            ForVaNetwork = forVaNetwork;

            var list = new List<Entity>();
            if (schemaList != null)
            {
                foreach (var item in schemaList)
                {
                    var child = await MapUpdate(item, parent);
                    if (child != null)
                    {
                        list.Add(child);
                        ModifyEntityList.Add(child);
                    }
                }

                if (!skipProviderRelationship && ModifyEntityList.Count > 0 && parent.GetType() == typeof(Account)) AddChildrenToProvider(list, parent);
            }

            return list;
        }

        public IEnumerable<SetStateRequest> DeleteMapperEntity<T>(IList<T> schemaList, Entity entity)
        {
            return MapDelete(schemaList, entity);
        }

        public void AddModifiedEntity(Entity entity)
        {
            ModifyEntityList.Add(entity);
        }

        public void SetContext(PpmsContext context)
        {
            Context = context;
        }

        protected void AssociateRelatedEntities(Entity parent, Entity entity, string relationship)
        {
            AssociateRelatedEntities(parent, new List<Entity>() { entity }, relationship);
        }

        protected void AssociateRelatedEntities(Entity parent, IEnumerable<Entity> entities, string relationship)
        {
            if (entities != null)
            {
                foreach (var item in entities)
                {
                    Context.AddRelatedObject(parent, new Relationship(relationship), item);
                }
            }
        }
    }
}